<?php
namespace lexicon\system\worker;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\package\Package;
use wcf\data\package\PackageCache;
use wcf\system\exception\SystemException;
use wcf\system\request\LinkHandler;
use wcf\system\worker\AbstractWorker;
use wcf\system\WCF;
use wcf\util\ClassUtil;

/**
 * Worker implementation for updating any wcf object to detect lexicon entries again.
 *
 * @author	Peter Fruehwirt
 * @copyright	2009-2013 VieCode
 * @license	VieCode Commercial License <http://www.viecode.com/licence/commercial/>
 * @package	com.viecode.lexicon.worker
 * @category	Lexicon Worker
 * @subpackage	system.worker
 */
class RebuildApplicationObjectWorker extends AbstractWorker {
	/**
	 * @see	\wcf\system\worker\AbstractWorker::$limit
	 */
	protected $limit = 50;

	/**
	 * object data offsets
	 * @var array
	 */
	protected $offsets = array();

	/**
	 * @see	\wcf\system\worker\AbstractWorker::countObjects()
	 */
	protected function countObjects() {
		if ($this->count === null) {
			$this->count = 0;

			foreach (ObjectTypeCache::getInstance()->getObjectTypes('com.viecode.lexicon.worker.application') as $objectType) {
				$package = PackageCache::getInstance()->getPackageByIdentifier($objectType->package);
				if ($package == null) continue;
				if ($objectType->minVersion != null && Package::compareVersion($package->packageVersion, $objectType->minVersion, '<')) continue;
				if ($objectType->maxVersion != null && Package::compareVersion($package->packageVersion, $objectType->maxVersion, '>')) continue;

				$instance = $this->getWorkerInstance($objectType->worker);
				$objectCount = $instance->getObjectCount();

				// save data offset
				$this->offsets[$this->count] = $objectType;

				$padding = $this->limit - ($objectCount % $this->limit);
				if ($padding == $this->limit) $padding = 0;

				// raise counter
				$this->count += ($objectCount + $padding);
			}
		}
	}

	/**
	 * creates worker instance
	 *
	 * @param $class
	 * @return \lexicon\system\application\worker\IApplicationWorker
	 * @throws SystemException
	 */
	private function getWorkerInstance($class) {
		if (!class_exists($class) || !ClassUtil::isInstanceOf($class, 'lexicon\system\application\worker\IApplicationWorker')) {
			throw new SystemException("class '".$class."' does not implement the 'lexicon\\system\\application\\worker\\IApplicationWorker' interface.");
		}

		return new $class();
	}

	/**
	 * @see	\wcf\system\worker\IWorker::execute()
	 */
	public function execute() {
		// find current worker offset
		$offset = $start = $this->limit * $this->loopCount;
		while ($start > 0) {
			if (isset($this->offsets[$start])) break;
			$start--;
		}
		$instance = $this->getWorkerInstance($this->offsets[$start]->worker);
		$instance->execute($offset - $start, $this->limit);
	}

	/**
	 * @see	\wcf\system\worker\AbstractWorker::IWorker()
	 */
	public function validate() {
		WCF::getSession()->checkPermissions(array('admin.system.canRebuildData'));
	}

	/**
	 * @see	\wcf\system\worker\AbstractWorker::IWorker()
	 */
	public function getProceedURL() {
		return LinkHandler::getInstance()->getLink('Worker', array('application' => 'lexicon'));
	}
}
