<?php

namespace shop\acp\form;

use shop\data\order\Order;
use shop\data\order\OrderEditor;
use shop\data\order\OrderList;
use shop\system\shop\IdentifierManager;
use wcf\form\AbstractForm;
use wcf\form\AbstractFormBuilderForm;
use wcf\system\exception\NamedUserException;
use wcf\system\form\builder\container\FormContainer;
use wcf\system\form\builder\field\BooleanFormField;
use wcf\system\form\builder\TemplateFormNode;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;

/**
 * Coverts invoice numbers between different invoice generates
 *
 * @author        {COPYRIGHT_AUTHOR}
 * @copyright     {COPYRIGHT_COMPANY}
 * @license       {COPYRIGHT_LICENSE}
 * @package       VieCode\Shop\Acp\Form
 *
 */
class InvoiceConvertForm extends AbstractFormBuilderForm {
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = 'shop.acp.menu.link.finance.invoice.convert';
	
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = ['admin.shop.products.canManageOrder'];
	
	/**
	 * first affected invoice identifier
	 */
	const FIRST_AFFECTED_INVOICE_IDENTIFIER = 'IN-000001/22';
	
	/**
	 * shop release date
	 */
	const SHOP_8_RELEASE = 1644102000;
	
	// shop.acp.invoice.convert.info
	
	/**
	 * affected orders 
	 * @var OrderList
	 */
	public $orderList;
	
	/**
	 * new invoice identifiers
	 * @var string[] 
	 */
	public $identifiers = [
		'production' => [],
		'test' => []
	];
	
	/**
	 * @inheritDoc
	 */
	protected function createForm() {
		parent::createForm();
		
		// check if year-independent counter is available
		$statement = WCF::getDB()->prepare("
			SELECT counterValue, environment
				FROM    shop1_counter
				WHERE	identifier = ? 
					AND counterYear IS NULL
		");
		$statement->execute([ 
			'invoice' 
		]);
		
		$counterIndependent = [];
		while ($row = $statement->fetchArray()) {
			$counterIndependent[$row['environment']] = $row['counterValue'];
		}

		if (count($counterIndependent) !== 2) {
			throw new NamedUserException(WCF::getLanguage()->get('shop.acp.invoice.convert.error.notApplicable'));
		}
		
		$statement = WCF::getDB()->prepare("
			SELECT time, environment
				FROM    shop1_order
				WHERE	invoiceIdentifier = ?
					AND time > ?
		");
		$statement->execute([ 
			self::FIRST_AFFECTED_INVOICE_IDENTIFIER, 
			self::SHOP_8_RELEASE 
		]);
		
		$firstOrderTime = [
			'production' => TIME_NOW+1, // no order will satisfy this condition 
			'test' => TIME_NOW+1
		];
		while ($row = $statement->fetchArray()) {
			$firstOrderTime[$row['environment']] = $row['time'];
		}

		$invoiceCreationField = 'time';
		if (SHOP_CREATE_INVOICE_EVENT === 'activation') {
			$invoiceCreationField = 'activationDate';
		}
		
		// check affected orders
		$this->orderList = new OrderList();
		$this->orderList->getConditionBuilder()->add('(('.$invoiceCreationField.' >= ? AND environment = ? OR '.$invoiceCreationField.' >= ? AND environment = ?) AND invoiceIdentifier IS NOT NULL) OR isInvoiceConverted = 1', [
			$firstOrderTime['production'], 'production', 
			$firstOrderTime['test'], 'test'
		]);
		$this->orderList->sqlOrderBy = 'invoiceIdentifier ASC';
		
		$this->orderList->readObjects();
		
		$this->identifiers = [];
		foreach ($this->orderList->getObjects() as $order) {
			if (!$order->isInvoiceConverted && \preg_match('/^IN-(\d{6})\/22$/', $order->invoiceIdentifier, $matches)) {
				$this->identifiers[$order->orderID] = self::getIdentifier($counterIndependent[$order->environment] + \intval($matches[1]));
			}
		}
		
		$this->form->appendChildren([
			TemplateFormNode::create('orderList')
				->templateName('invoiceConvertOrderList')
				->application('shop')
				->variables([
					'orderList' => $this->orderList,
					'identifiers' => $this->identifiers
				]),
			
			FormContainer::create('confirmation')
				->label('shop.acp.invoice.convert.confirmation')
				->available(count($this->identifiers) === $this->orderList->count())
				->appendChildren([
					BooleanFormField::create('confirm')
						->label('shop.acp.invoice.convert.confirm')
						->required(count($this->identifiers) === $this->orderList->count())
						->value(0)
				])
		]);
		
		if (count($this->identifiers) !== $this->orderList->count()) {
			$this->form->addDefaultButton(false);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		AbstractForm::save();
	
		foreach ($this->identifiers as $orderID => $invoiceIdentifier) {
			$order = $this->orderList->getObjects()[$orderID];
			
			$editor = new OrderEditor($order);
			$editor->update([
				'invoiceIdentifier' => $invoiceIdentifier,
				'originalInvoiceIdentifier' => $order->invoiceIdentifier,
				'isInvoiceConverted' => 1,
				'converterCancellationInvoiceIdentifier' => $this->getCancellationIdentifier($order),
				'converterCancellationTime' => TIME_NOW
			]);
			
			// increase counter
			IdentifierManager::next('invoice', false, $order->environment);
			
			// regenerate invoice
			$editor->getInvoicePDFDocument(true);
		}
		
		// delete year dependent invoice counter
		$statement = WCF::getDB()->prepare("
			DELETE FROM	shop1_counter
				WHERE	identifier = ?
				AND 	counterYear = ?
		");
		$statement->execute([ 
			'invoice', 
			'2022' 
		]);
		
		// set legacy invoice class
		$statement = WCF::getDB()->prepare("
			UPDATE	wcf1_option
				SET	optionValue = ?
				WHERE	optionName = ?
		");
		$statement->execute([
			'shop\system\invoice\identifier\LegacyInvoiceIdentifier',
			'shop_invoice_format_class'
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function saved() {
		parent::saved();
		
		// reload page
		HeaderUtil::redirect(LinkHandler::getInstance()->getLink('InvoiceConvert', [
			'application' => 'shop'
		]));
		exit;
	}
	
	/**
	 * returns invoice identifier 
	 * 
	 * @param int $counter
	 * @return string
	 */
	private static function getIdentifier(int $counter) {
		return 'IN-' . \str_pad($counter, 6, '0', STR_PAD_LEFT) . '/22' ;
	}
	
	/**
	 * returns cancellation identifier
	 * 
	 * @param Order $order
	 * @return string
	 */
	public function getCancellationIdentifier(Order $order) {
		$date = \getdate($order->time);
		
		return 'ST-' . \str_pad(IdentifierManager::next('converterCancellation', true, $order->environment), 6, '0', STR_PAD_LEFT) . '/' . \substr($date['year'], 2, 2);
	}
}
