<?php

namespace shop\acp\action;

use Dompdf\Dompdf;
use shop\data\order\Order;
use shop\system\cache\runtime\OrderRuntimeCache;
use shop\system\notification\NotificationManager;
use shop\system\shop\TaxManager;
use shop\system\template\PDFTemplateEngine;
use wcf\action\AbstractAction;
use wcf\system\event\EventHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;
use wcf\util\FileReader;

/**
 * order file download
 *
 * @author        {COPYRIGHT_AUTHOR}
 * @copyright     {COPYRIGHT_COMPANY}
 * @license       {COPYRIGHT_LICENSE}
 * @package       VieCode\Shop\Acp\Action
 */
class InvoiceConvertCancellationDownloadAction extends AbstractAction {
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = ['admin.shop.products.canManageOrder'];
	
	/**
	 * order id
	 *
	 * @var integer
	 */
	public $orderID;
	
	/**
	 * order object
	 *
	 * @var Order
	 */
	public $order;
	
	/**
	 * @var FileReader
	 */
	public $fileReader;
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		if (isset($_REQUEST['id'])) {
			$this->orderID = (int) $_REQUEST['id'];
			$this->order = OrderRuntimeCache::getInstance()->getObject($this->orderID);
		}
		
		if ($this->order === null || !$this->order->getObjectID() || !$this->order->converterCancellationInvoiceIdentifier) {
			throw new IllegalLinkException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function execute() {
		parent::execute();
		
		$path = SHOP_DIR . 'storage/converterCancellation/' . $this->order->getObjectID() . '_' . $this->order->time . '.pdf';
		
		// get payment information text
		$paymentInformationText = "";
		$paymentType = $this->order->getPaymentType();
		if ($paymentType !== null && $paymentType->getPaymentInvoiceInformation()) {
			// get language value and compiled code
			$scriptingCompiler = LanguageFactory::getInstance()->getScriptingCompiler();
			$compiled = [
				'paymentText' => $scriptingCompiler->compileString(
					'paymentInformationText@converterCancellation-' . $this->order->orderID,
					WCF::getLanguage()->get($paymentType->getPaymentInvoiceInformation())
				),
			];
			
			// generate payment text content using a template engine and language variables
			$paymentInformationText = WCF::getTPL()->fetchString($compiled['paymentText']['template'], [
				'template' => 'converterCancellation',
				'order' => $this->order,
			]);
		}
		
		$smallBusinessInformation = SHOP_SMALLBUSINESS ? TaxManager::getLegalInformation(SHOP_SMALLBUSINESS_LEGAL, $this->order->getInvoiceAddress()->countryCode) : null;
		
		
		// set language ID
		PDFTemplateEngine::getInstance()->setLanguageID(WCF::getLanguage()->languageID);
		
		// assign objects for language variables
		WCF::getTPL()->assign([
			'template' => 'converterCancellation',
			'order' => $this->order,
			'invoiceAddress' => $this->order->getInvoiceAddress(),
			'tax' => $this->order->getTaxRates(),
		]);
		
		// load dompdf library
		require_once(WCF_DIR . 'lib/system/api/dompdf/autoload.php');
		
		$pdf = new Dompdf();
		$pdf->getOptions()->setChroot(SHOP_DIR);
		$pdf->getOptions()->setTempDir(WCF_DIR . 'tmp/');
		$pdf->getOptions()->setLogOutputFile(WCF_DIR . 'tmp/dompdf.log');
		
		$parameters = [
			'dompdf' => clone $pdf,
			'variables' => [
				'template' => 'converterCancellation',
				'order' => $this->order,
				'invoiceAddress' => $this->order->getInvoiceAddress(),
				'tax' => $this->order->getTaxRates(),
				'css' => file_get_contents(file_exists(SHOP_DIR . 'storage/invoice-custom.css') ? SHOP_DIR . 'storage/invoice-custom.css' : SHOP_DIR . 'storage/invoice.css'),
				'path' => SHOP_DIR,
				'paymentInformationText' => $paymentInformationText,
				'smallBusinessInformation' => $smallBusinessInformation,
			],
		];
		EventHandler::getInstance()->fireAction($this, 'setVariables', $parameters);
		$pdf = $parameters['pdf'] ?? $pdf;
		
		$pdf->loadHtml(PDFTemplateEngine::getInstance()->fetch('converterCancellation', 'shop@pdf', $parameters['variables']));
		$pdf->render();
		$output = $pdf->output();
		
		\file_put_contents($path, $output);
		
		$this->fileReader = new FileReader($path, [
			'filename' => NotificationManager::sanitizeFileName($this->order->converterCancellationInvoiceIdentifier . '.pdf'),
			'mimeType' => 'application/pdf',
			'filesize' => \filesize($path),
			'showInline' => true,
			'enableRangeSupport' => false,
			'expirationDate' => TIME_NOW + 31536000,
			'maxAge' => 31536000,
		]);
		
		$this->executed();
	}
	
	/**
	 * @inheritDoc
	 */
	protected function executed() {
		parent::executed();
		
		$this->fileReader->send();
		exit;
	}
}
